#!/bin/bash

# BerqWP PHP-Scoper Build Script
# This script creates a scoped version of vendor dependencies to avoid conflicts

set -e

echo "==========================================="
echo "BerqWP PHP-Scoper Build Process"
echo "==========================================="

# Colors for output
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Get script directory
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
cd "$SCRIPT_DIR"

echo -e "${YELLOW}Step 1: Checking dependencies...${NC}"
if [ ! -d "vendor" ]; then
    echo -e "${RED}Error: vendor directory not found. Run 'composer install' first.${NC}"
    exit 1
fi

if [ ! -f "php-scoper.phar" ]; then
    echo -e "${RED}Error: php-scoper.phar not found in current directory.${NC}"
    exit 1
fi

echo -e "${GREEN}✓ Dependencies found${NC}"

echo -e "${YELLOW}Step 2: Cleaning previous build...${NC}"
if [ -d "vendor-scoped" ]; then
    rm -rf vendor-scoped
    echo -e "${GREEN}✓ Cleaned vendor-scoped directory${NC}"
fi

echo -e "${YELLOW}Step 3: Running PHP-Scoper...${NC}"
php php-scoper.phar add-prefix \
    --config=scoper.inc.php \
    --output-dir=vendor-scoped \
    --force \
    --no-interaction

if [ $? -ne 0 ]; then
    echo -e "${RED}Error: PHP-Scoper failed${NC}"
    exit 1
fi

echo -e "${GREEN}✓ PHP-Scoper completed successfully${NC}"

echo -e "${YELLOW}Step 4: Creating vendor-scoped/autoload.php wrapper...${NC}"
cat > vendor-scoped/autoload.php << 'EOF'
<?php

// BerqWP Scoped Autoloader
// This file loads the scoped vendor dependencies with BerqWP\GuzzleHttp\ prefix

// Load the scoped autoloader generated by PHP-Scoper
if (file_exists(__DIR__ . '/scoper-autoload.php')) {
    require_once __DIR__ . '/scoper-autoload.php';
}

// Load Composer's autoloader
if (file_exists(__DIR__ . '/autoload_real.php')) {
    require_once __DIR__ . '/autoload_real.php';
}

// Try to get the Composer loader
$composerAutoloaderClasses = get_declared_classes();
foreach ($composerAutoloaderClasses as $class) {
    if (strpos($class, 'ComposerAutoloaderInit') === 0) {
        if (method_exists($class, 'getLoader')) {
            return $class::getLoader();
        }
    }
}

// Fallback: just return true
return true;
EOF

echo -e "${GREEN}✓ Autoloader wrapper created${NC}"

echo -e "${YELLOW}Step 5: Verifying scoped classes...${NC}"

# Check if critical classes exist
if grep -r "namespace BerqWP\\\\GuzzleHttp" vendor-scoped/ > /dev/null; then
    echo -e "${GREEN}✓ Scoped namespace 'BerqWP\\GuzzleHttp' found${NC}"
else
    echo -e "${RED}Error: Scoped namespace not found!${NC}"
    exit 1
fi

# Check for unscoped GuzzleHttp references (should not exist)
if grep -r "^namespace GuzzleHttp" vendor-scoped/ > /dev/null; then
    echo -e "${RED}Warning: Found unscoped 'namespace GuzzleHttp' references${NC}"
    echo "This might indicate scoping didn't work properly."
else
    echo -e "${GREEN}✓ No unscoped GuzzleHttp namespaces found${NC}"
fi

echo ""
echo "==========================================="
echo -e "${GREEN}Build completed successfully!${NC}"
echo "==========================================="
echo ""
echo "Scoped vendor directory: vendor-scoped/"
echo "Next steps:"
echo "1. Update import statements in src/ files"
echo "2. Update berqwp.php to use vendor-scoped/autoload.php"
echo "3. Test the plugin with scoped dependencies"
echo ""
